<?php
declare(strict_types=1);

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../app/security.php';

tpcms_start_session_secure();
tpcms_send_security_headers();

$__minPhp = '8.0';  // 推奨バージョン（必要なら変更可）
$__phpTooLow = function_exists('version_compare') ? version_compare(PHP_VERSION, $__minPhp, '<') : true;
// ※ 非ブロッキング。古いPHPでも登録は続行できるが、下で注意を表示します。

$adminFile = __DIR__ . '/../data/admin.json';

// すでに登録済みならログイン画面へ
if (is_file($adminFile)) {
    header('Location: ./login.php');
    exit;
}

$error = '';
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'POST') {
    tpcms_require_post_csrf();

    $email = trim((string)($_POST['email'] ?? ''));
    $pass  = (string)($_POST['pass']  ?? '');
    $pass2 = (string)($_POST['pass2'] ?? '');

    // 入力バリデーション
    if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'メールアドレスを正しく入力してください。';
    } elseif (strlen($pass) < 8) {
        $error = 'パスワードは8文字以上で入力してください。';
    } elseif (!hash_equals($pass, $pass2)) {
        $error = '確認用パスワードが一致しません。';
    } else {
        // 二重作成防止（再チェック）
        if (is_file($adminFile)) {
            header('Location: ./login.php');
            exit;
        }

        // データ作成
        $dir = dirname($adminFile);
        if (!is_dir($dir)) {
            @mkdir($dir, 0755, true);
        }

        $payload = [
            'email'         => $email,
            'password_hash' => password_hash($pass, PASSWORD_DEFAULT),
            'created_at'    => date('c'),
        ];

        $json = json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
        $ok   = false;
        if ($json !== false) {
            $tmp = $adminFile . '.tmp';
            if (@file_put_contents($tmp, $json, LOCK_EX) !== false) {
                $ok = @rename($tmp, $adminFile);
                @chmod($adminFile, 0600); // 可能なら読み書き最小化
            } else {
                @unlink($tmp);
            }
        }

        if ($ok) {
            // 登録直後はそのままログイン扱いにして管理トップへ
            $_SESSION['admin_logged_in'] = true;
            $_SESSION['admin_user']      = $email;
            tpcms_session_regenerate_on_login();

            header('Location: ./index.php');
            exit;
        } else {
            $error = '登録に失敗しました。/data の書き込み権限を確認してください。';
        }
    }
}

$token = tpcms_csrf_token();
?>
<!doctype html>
<html lang="ja">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>初回セットアップ（管理者登録）｜Template Party CMS</title>
  <link rel="stylesheet" href="./assets/admin.css">
</head>
<body class="login">

<div id="container">

  <h1>初回セットアップ（管理者登録）</h1>

  <?php if ($__phpTooLow): ?>
    <p class="notice">
      <strong style="color: red;">※現在の環境ではCMSが動作しない可能性があります。</strong><br>
      あなたの現在のPHPは<strong style="color: red;"><?= tpcms_h(PHP_VERSION) ?></strong>です。<br>
      正常に動作しない場合はサーバーのコントロールパネルでPHPを最低でも<strong style="color: red;"><?= tpcms_h($__minPhp) ?></strong>（推奨は <strong style="color: red;">8.2〜8.3</strong>）へ切り替えて下さい。<br>
      尚、他にphpを使ったシステムがあるとそちらで不具合が出る可能性が高いので切り替えの際は注意して下さい。
    </p>
  <?php endif; ?>

  <?php if ($error): ?>
    <p class="error"><?= tpcms_h($error) ?></p>
  <?php endif; ?>

  <form method="post" action="./setup_admin.php">
    <input type="hidden" name="_csrf" value="<?= tpcms_h($token) ?>">

    <p>
      <label>メールアドレス
        <input type="email" name="email" required>
      </label>
    </p>
    <p>
      <label>パスワード（8文字以上）
        <input type="password" name="pass" required minlength="8">
      </label>
    </p>
    <p>
      <label>パスワード（確認）
        <input type="password" name="pass2" required minlength="8">
      </label>
    </p>
    <p><button type="submit" class="btn1">登録して管理トップへ</button></p>
    <p class="muted">※登録済みの場合は <a href="./login.php">ログイン</a> へ。</p>
  </form>

</div>

</body>
</html>
