<?php
declare(strict_types=1);

require_once __DIR__ . '/_auth.php';
tpcms_require_admin();

require_once __DIR__ . '/../config.php';

// --- JSONユーティリティ（存在しなければ定義） ---
if (!function_exists('tpcms_json_read')) {
    function tpcms_json_read(string $path, $default = []) {
        if (is_file($path)) {
            $json = file_get_contents($path);
            $data = json_decode((string)$json, true);
            if (is_array($data) || is_object($data)) return $data;
        }
        return $default;
    }
}
if (!function_exists('tpcms_json_write')) {
    function tpcms_json_write(string $path, $data): bool {
        $dir = dirname($path);
        if (!is_dir($dir) && !@mkdir($dir, 0777, true)) return false;
        $json = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
        if ($json === false) return false;
        $tmp = $path . '.tmp';
        if (@file_put_contents($tmp, $json) === false) return false;
        @chmod($tmp, 0664);
        $ok = @rename($tmp, $path);
        if (!$ok) { @unlink($tmp); return false; }
        @chmod($path, 0664);
        clearstatcache(true, $path);
        return true;
    }
}

$path     = __DIR__ . '/../data/license.json';
$default  = ['code' => ''];
$current  = tpcms_json_read($path, $default);
if (!is_array($current)) $current = $default;

$notice = null; $error = null;

// --- 保存処理 ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
tpcms_require_post_csrf();
    $code = '';
    if (isset($_POST['license']['code']) && is_scalar($_POST['license']['code'])) {
        $code = trim((string)$_POST['license']['code']);
    }

    // 空文字は許可（空なら著作表記を表示）
    if ($code === '') {
        $updated = $current;
        $updated['code'] = '';
        if (tpcms_json_write($path, $updated)) {
            $current = $updated;
            $notice = '保存しました。';
        } else {
            $error = '保存に失敗しました。権限やディスク容量を確認してください。';
        }
    } else {
        // 長さガード（念のため）
        if (mb_strlen($code, 'UTF-8') > 128) {
            $error = 'ライセンスコードが長すぎます（最大128文字）。';
        }
        // 書式チェック：
        //   ・先頭は <!--License  または <!--eLicense
        //   ・続けて 8桁の数字（YYYYMMDD）
        //   ・続けて 英数字5桁（大小どちらでもOK）
        //   ・末尾は -->
        elseif (!preg_match('/^<!--(?:e?License)\d{8}[A-Za-z0-9]{5}-->$/', $code)) {
            $error = 'ライセンスコードの形式が不正です。';
        } else {
            $updated = $current;
            $updated['code'] = $code;

            if (tpcms_json_write($path, $updated)) {
                $current = $updated;
                $notice = '保存しました。';
            } else {
                $error = '保存に失敗しました。権限やディスク容量を確認してください。';
            }
        }
    }
}

// 表示用
$code = is_scalar($current['code'] ?? null) ? (string)$current['code'] : '';

?><!doctype html>
<html lang="ja">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>ライセンス｜Template Party CMS</title>
  <link rel="stylesheet" href="./assets/admin.css">
</head>
<body class="admin">

<header>
<div>
<a href="./">ダッシュボード</a>
<a href="../" target="_blank" rel="noopener">公開側トップ</a>
</div>
<div>
<a href="./logout.php">ログアウト</a>
</div>
</header>

  <div id="container">
    <h1>ライセンス管理</h1>

    <?php if ($notice): ?>
      <div class="alert success"><?= tpcms_h($notice) ?></div>
    <?php endif; ?>
    <?php if ($error): ?>
      <div class="alert error"><?= tpcms_h($error) ?></div>
    <?php endif; ?>

    <div class="card">
      <form method="post" action="./license.php">
      <input type="hidden" name="_csrf" value="<?= tpcms_h(tpcms_csrf_token()) ?>">

        <div class="row">
          <label class="label">ライセンスコード</label>
          <input type="text" name="license[code]" value="<?= tpcms_h($code) ?>" placeholder="ここにライセンスコードを入力" maxlength="128">
        </div>

        <!-- 下部の保存 -->
        <div class="btnbar">
          <button type="submit" class="btn1">保存</button>
        </div>
      </form>
    </div>
  </div>

  <script src="./assets/admin.js"></script>
</body>
</html>
