<?php
declare(strict_types=1);

/**
 * tpcms-form 定義ユーティリティ
 * - partial 先頭の <!-- tpcms-form: {...} --> から定義を取得
 * - 管理UIの自動生成で利用するための最小スキーマ整形と help_html サニタイズ
 *
 * 提供関数:
 *   - tpcms_formdef_load_by_name(string $partialName): ?array
 *   - tpcms_formdef_parse_from_html(string $html): ?array
 *   - tpcms_formdef_sanitize_help_html(string $html): string
 */

if (!function_exists('tpcms_formdef_load_by_name')) {
    /**
     * アクティブテーマの partial ファイル名から tpcms-form 定義配列を取得。
     * 例: tpcms_formdef_load_by_name('cards.html')
     */
    function tpcms_formdef_load_by_name(string $partialName): ?array {
        $root = dirname(__DIR__); // /app の1つ上＝プロジェクトルート

        $activePath = $root . '/themes/_active.json';
        if (!is_file($activePath)) return null;

        $active = json_decode((string)@file_get_contents($activePath), true);
        if (!is_array($active) || empty($active['theme']) || empty($active['color'])) {
            return null;
        }

        $path = $root . '/themes/' . $active['theme'] . '/' . $active['color'] . '/partials/' . $partialName;
        if (!is_file($path)) return null;

        $html = (string)@file_get_contents($path);
        if ($html === '') return null;

        return tpcms_formdef_parse_from_html($html) ?: null;
    }
}

if (!function_exists('tpcms_formdef_parse_from_html')) {
    /**
     * partial HTML 文字列から <!-- tpcms-form: {...} --> を抽出して配列化。
     * help_html は許可タグのみ通過。fields/repeaters は配列を保証。
     */
    function tpcms_formdef_parse_from_html(string $html): ?array {
        if (!preg_match('/<!--\s*tpcms-form\s*:\s*(\{.*\})\s*-->/si', $html, $m)) {
            return null; // 定義なし
        }
        $json = $m[1];
        $def = json_decode($json, true);
        if (!is_array($def)) return null;

        // 最低限の正規化
        $def['label']     = isset($def['label']) ? (string)$def['label'] : '';
        $def['category']  = isset($def['category']) ? (string)$def['category'] : '未分類';

        if (isset($def['help_html'])) {
            $def['help_html'] = tpcms_formdef_sanitize_help_html((string)$def['help_html']);
        }

        $def['fields']    = isset($def['fields']) ? (array)$def['fields'] : [];
        $def['repeaters'] = isset($def['repeaters']) ? (array)$def['repeaters'] : [];

        // 配列要素のみ残す（数値キー配列に詰め直し）
        $def['fields']    = array_values(array_filter($def['fields'], 'is_array'));
        $def['repeaters'] = array_values(array_filter($def['repeaters'], 'is_array'));

        // 各要素を配列にキャスト（UI側で取り扱いやすくする）
        foreach (['fields', 'repeaters'] as $k) {
            foreach ($def[$k] as $i => $row) {
                $def[$k][$i] = (array)$row;
            }
        }
        return $def;
    }
}

if (!function_exists('tpcms_formdef_sanitize_help_html')) {
    /**
     * help_html の簡易サニタイズ
     * 許可タグ: a, strong, em, code, ul, ol, li
     * a[href] が javascript:, data:, vbscript:, file: で始まる場合は "#" に差し替え。
     */
    function tpcms_formdef_sanitize_help_html(string $html): string {
        $html = strip_tags($html, '<a><strong><em><code><ul><ol><li>');

        // aタグの href を検査して危険スキームを無効化
        $html = preg_replace_callback('/<a\b([^>]*)>/i', function ($m) {
            $attrs = $m[1];

            if (preg_match('/href\s*=\s*(["\']?)([^"\'>\s]+)\1/i', $attrs, $mm)) {
                $href = $mm[2];
                if (preg_match('~^(?:javascript|data|vbscript|file):~i', $href)) {
                    $attrs = preg_replace('/href\s*=\s*(["\']?)[^"\'>\s]+\1/i', 'href="#"', $attrs);
                }
            }
            return '<a' . $attrs . '>';
        }, $html);

        return $html;
    }
}
