<?php
declare(strict_types=1);

require_once __DIR__ . '/_auth.php';
tpcms_require_admin();

require_once __DIR__ . '/../config.php';

// site.php と同等のユーティリティ（存在しなければ定義）
if (!function_exists('tpcms_json_read')) {
    function tpcms_json_read(string $path, $default = []) {
        if (is_file($path)) {
            $json = file_get_contents($path);
            $data = json_decode((string)$json, true);
            if (is_array($data) || is_object($data)) return $data;
        }
        return $default;
    }
}
if (!function_exists('tpcms_json_write')) {
    function tpcms_json_write(string $path, $data): bool {
        $dir = dirname($path);
        if (!is_dir($dir) && !@mkdir($dir, 0777, true)) {
            return false;
        }
        $json = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
        if ($json === false) return false;

        $tmp = $path . '.tmp';
        if (@file_put_contents($tmp, $json) === false) return false;
        @chmod($tmp, 0664);
        $ok = @rename($tmp, $path);
        if (!$ok) { @unlink($tmp); return false; }
        @chmod($path, 0664);
        clearstatcache(true, $path);
        return true;
    }
}

// ------------------------------------------------------------
// 現在値の読込（既定形を与えつつ未知キーは温存）
// ------------------------------------------------------------
$contactPath = __DIR__ . '/../data/contact.json';

// 仕様の既定形（最小3項目）
// 参考：STEP8/データ仕様（to_email/subject/fields[3]）
$defaultContact = [
    'to_email' => '',
    'subject'  => 'お問い合わせ',
    'fields'   => [
        ['key' => 'name',    'label' => 'お名前',           'type' => 'text',     'required' => true],
        ['key' => 'email',   'label' => 'メール',           'type' => 'text',     'required' => true],
        ['key' => 'message', 'label' => 'お問い合わせ内容', 'type' => 'textarea', 'required' => true],
    ],
    // 追加（新規キー／未知キーは従来どおり温存）
    'auto_reply' => false,
    'site_name'  => '',
    'header_msg' => '以下の内容でお問い合わせを受け付けました。',
    'footer_msg' => "――――――――――――――――\nサンプル署名（差し替えてください）\n株式会社○○○○\nTEL: 00-0000-0000\nE-mail: info@example.com",
];

$current = tpcms_json_read($contactPath, $defaultContact);
if (!is_array($current)) $current = $defaultContact;

// ------------------------------------------------------------
// POST（保存）
// ------------------------------------------------------------
$notice = null; $error = null;

// POST: 保存
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    tpcms_require_post_csrf();

    // 既存JSONの読み込み（上部で $contactPath / $current が定義済み）
    $src = is_array($current) ? $current : [];
    $existingFields = (isset($src['fields']) && is_array($src['fields'])) ? $src['fields'] : [];

    // 入力は contact[...] 階層で来る
    $posted = $_POST['contact'] ?? [];

    $toEmail   = is_scalar($posted['to_email']  ?? null) ? trim((string)$posted['to_email'])  : '';
    $subject   = is_scalar($posted['subject']   ?? null) ? trim((string)$posted['subject'])   : '';
    $siteName  = is_scalar($posted['site_name'] ?? null) ? trim((string)$posted['site_name']) : '';
    $autoReply = !empty($posted['auto_reply']); // checkbox
    $headerMsg = is_scalar($posted['header_msg'] ?? null) ? (string)$posted['header_msg'] : '';
    $footerMsg = is_scalar($posted['footer_msg'] ?? null) ? (string)$posted['footer_msg'] : '';

    // 簡易バリデーション
    $errors = [];
    if ($subject === '') {
        $errors[] = '件名（subject）は必須です。';
    }
    if ($toEmail !== '' && !filter_var($toEmail, FILTER_VALIDATE_EMAIL)) {
        $errors[] = '送信先メールアドレス（to_email）の形式が正しくありません。';
    }

    if ($errors) {
        $error = implode(' / ', $errors);
    } else {
        // 既存値ベースで未知キー温存。編集したキーのみ上書き
        $updated = $src;
        $updated['to_email']   = $toEmail;
        $updated['subject']    = $subject;
        $updated['site_name']  = $siteName;
        $updated['auto_reply'] = $autoReply;
        $updated['header_msg'] = $headerMsg;
        $updated['footer_msg'] = $footerMsg;

        // fields[] は常に現状維持（上書きしない）
        $updated['fields'] = $existingFields;

        if (tpcms_json_write($contactPath, $updated)) {
            $current = $updated;
            $notice = '保存しました。';
        } else {
            $error = '保存に失敗しました。';
        }
    }
}

// 表示用の安全値
$to        = is_scalar($current['to_email']   ?? null) ? (string)$current['to_email']   : '';
$subj      = is_scalar($current['subject']    ?? null) ? (string)$current['subject']    : '';
$siteName  = is_scalar($current['site_name']  ?? null) ? (string)$current['site_name']  : '';
$autoReply = !empty($current['auto_reply']);
$headerMsg = is_scalar($current['header_msg'] ?? null) ? (string)$current['header_msg'] : '';
$footerMsg = is_scalar($current['footer_msg'] ?? null) ? (string)$current['footer_msg'] : '';

$lab = [
    'name'    => 'お名前',
    'email'   => 'メール',
    'message' => 'お問い合わせ内容',
];
foreach (['name','email','message'] as $k) {
    $cur = $current['fields'] ?? [];
    $found = null;
    if (is_array($cur)) {
        foreach ($cur as $f) {
            if (is_array($f) && ($f['key'] ?? null) === $k) { $found = $f; break; }
        }
    }
    if (is_array($found) && is_scalar($found['label'] ?? null)) {
        $lab[$k] = (string)$found['label'];
    }
}

?><!doctype html>
<html lang="ja">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>お問い合わせ設定｜Template Party CMS</title>
  <link rel="stylesheet" href="./assets/admin.css">
</head>
<body class="admin contact-settings">

<header>
<div>
<a href="./">ダッシュボード</a>
<a href="../" target="_blank" rel="noopener">公開側トップ</a>
</div>
<div>
<a href="./logout.php">ログアウト</a>
</div>
</header>

  <div id="container">
  
    <h1>お問い合わせ設定</h1>

    <?php if ($notice): ?>
      <div class="notice"><?= tpcms_h($notice) ?></div>
    <?php endif; ?>
    <?php if ($error): ?>
      <div class="err"><?= tpcms_h($error) ?></div>
    <?php endif; ?>

    <div class="card">
      <form method="post" action="./contact_settings.php">
      <input type="hidden" name="_csrf" value="<?= tpcms_h(tpcms_csrf_token()) ?>">
        <!-- 上部の保存 -->

        <div class="row mb3rem">
          <label>
            <input type="checkbox" name="contact[auto_reply]" value="1" <?= $autoReply ? 'checked' : '' ?>> 自動返信機能を使う
          </label>
        </div>

        <div class="row">
          <label class="label">送信先メールアドレス<span class="muted">（必須）</span></label>
          <input type="text" name="contact[to_email]" value="<?= tpcms_h($to) ?>" required>
        </div>

        <div class="row">
          <label class="label">件名<span class="muted">（必須）</span></label>
          <input type="text" name="contact[subject]" value="<?= tpcms_h($subj) ?>" required>
        </div>

        <div class="row">
          <label class="label">サイト名</label>
          <input type="text" name="contact[site_name]" value="<?= tpcms_h($siteName) ?>" placeholder="例：サンプルカンパニー">
        </div>

        <div class="row">
          <label class="label">ヘッダーメッセージ</label>
          <textarea name="contact[header_msg]" rows="3"><?= tpcms_h($headerMsg) ?></textarea>
        </div>

        <div class="row">
          <label class="label">フッターメッセージ（署名など）</label>
          <textarea name="contact[footer_msg]" rows="4"><?= tpcms_h($footerMsg) ?></textarea>
        </div>

        <!-- 下部の保存 -->
        <div class="btnbar">
          <button type="submit" class="btn1">保存</button>
        </div>
      </form>
    </div>
  </div>

  <script src="./assets/admin.js"></script>
</body>
</html>
