<?php
declare(strict_types=1);

require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth/_auth.php';
require_once __DIR__ . '/../security/_csrf.php';

// 未ログインならログインへ
auth_require_login();

require_once __DIR__ . '/../app/categories.php';

// --- 並び順（ord）を保証：無ければ追加し、現行の新しい順のまま 1..N 採番 ---
try {
    $pdo = db();
    $hasOrd = false;
    foreach ($pdo->query("PRAGMA table_info('categories')") as $col) {
        if ((string)($col['name'] ?? '') === 'ord') { $hasOrd = true; break; }
    }
    if (!$hasOrd) {
        $pdo->beginTransaction();
        $pdo->exec('ALTER TABLE categories ADD COLUMN ord INTEGER');
        // 既存UIと同じ「新しい順（created_at DESC, id DESC）」で初期採番
        $ids = $pdo->query('SELECT id FROM categories ORDER BY created_at DESC, id DESC')->fetchAll(PDO::FETCH_COLUMN);
        $up  = $pdo->prepare('UPDATE categories SET ord = ? WHERE id = ?');
        $ord = 1;
        foreach ($ids as $cid) { $up->execute([$ord++, (int)$cid]); }
        $pdo->commit();
    }
} catch (Throwable $e) {
    // 失敗時は created_at DESC を継続（致命停止はしない）
}

// --- 追加/削除/公開切替処理（POST） ---
$error = '';
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'POST') {
    csrf_check_or_die();
    $action = (string)($_POST['_action'] ?? '');
    try {
        if ($action === 'create') {
            $name      = trim((string)($_POST['name'] ?? ''));
            $slug      = trim((string)($_POST['slug'] ?? ''));
            $is_active = isset($_POST['is_active']);
            tpcms_cat_insert($name, $slug, $is_active);
            header('Location: ./categories.php?ok=1');
            exit;
        } elseif ($action === 'delete') {
            $id = (int)($_POST['id'] ?? 0);
            if ($id <= 0) {
                throw new InvalidArgumentException('Invalid ID.');
            }
            tpcms_cat_delete($id);
            header('Location: ./categories.php?deleted=1');
            exit;
        } elseif ($action === 'toggle') {
            $id = (int)($_POST['id'] ?? 0);
            $to = (int)($_POST['to_active'] ?? -1);
            if ($id <= 0 || ($to !== 0 && $to !== 1)) {
                throw new InvalidArgumentException('Invalid toggle request.');
            }
            tpcms_cat_set_active($id, $to === 1);
            header('Location: ./categories.php?toggled=1');
            exit;
        } elseif ($action === 'sort') {
            // ☰ドラッグ後に送られてくる ids[] の順で ord を 1..N に再採番
            $ids = $_POST['ids'] ?? [];
            if (is_array($ids) && !empty($ids)) {
                $ids = array_values(array_filter(array_map(function ($v) {
                    $s = is_array($v) ? reset($v) : $v;
                    return preg_match('/^\d+$/', (string)$s) ? (int)$s : null;
                }, $ids), fn($v) => $v !== null));

                if (!empty($ids)) {
                    $pdo = db();
                    $pdo->beginTransaction();
                    $up  = $pdo->prepare('UPDATE categories SET ord = ? WHERE id = ?');
                    $ord = 1;
                    foreach ($ids as $cid) { $up->execute([$ord++, (int)$cid]); }
                    $pdo->commit();
                }
            }
            header('Content-Type: application/json; charset=UTF-8');
            echo '{"ok":true}';
            exit;
        }
    } catch (Throwable $e) {
        $msg = $e->getMessage();
        if ($e instanceof PDOException && $e->getCode() === '23000') {
            $error = 'このスラッグは既に使用されています。別のスラッグを指定してください。';
        } elseif (stripos($msg, 'UNIQUE') !== false && stripos($msg, 'slug') !== false) {
            $error = 'このスラッグは既に使用されています。別のスラッグを指定してください。';
        } else {
            $error = $msg;
        }
    }
}

// --- 一覧取得（常に最後に呼ぶ）※検索/ソート/フィルタは撤去
$stmt = db()->query("SELECT * FROM categories ORDER BY COALESCE(ord, 999999) ASC, id ASC");
$cats = $stmt->fetchAll();
$cnt  = count($cats);
?><!doctype html>
<html lang="ja">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>カテゴリ管理｜Template Party CMS</title>
  <link rel="stylesheet" href="<?= h(asset_url('/admin/assets/admin.css', true)) ?>">
</head>
<body class="admin categories">
<?php include __DIR__ . '/_header.php'; ?>

<div id="container">
  <h1>カテゴリ管理</h1>

  <?php if ($error !== ''): ?>
    <div class="admin-alert error"><?= h($error) ?></div>
  <?php elseif (isset($_GET['ok'])): ?>
    <div class="admin-alert success">カテゴリを追加しました。</div>
  <?php elseif (isset($_GET['updated'])): ?>
    <div class="admin-alert success">カテゴリを更新しました。</div>
  <?php elseif (isset($_GET['deleted'])): ?>
    <div class="admin-alert success">カテゴリを削除しました。</div>
  <?php elseif (isset($_GET['toggled'])): ?>
    <div class="admin-alert success">公開設定を更新しました。</div>
  <?php endif; ?>

  <h2>カテゴリ登録</h2>

  <div class="row">
    <form method="post" action="./categories.php">
    <?= csrf_input_tag() ?>
      <input type="hidden" name="_action" value="create">
      <div class="row">
        <label class="label">カテゴリ名</label>
        <input type="text" name="name" required placeholder="例）賃貸物件">
      </div>
      <div class="row">
        <label class="label">スラッグ</label>
        <input type="text" name="slug" required pattern="^[A-Za-z0-9_-]+$" placeholder="例）chintai">
        <p class="small">英数字・ハイフン・アンダースコアのみ（例：news / jobs / cat_01）</p>
      </div>
      <div class="row">
        <label class="checkbox"><input type="checkbox" name="is_active" checked> 公開する</label>
      </div>
      <div class="row">
        <button type="submit" class="btn1">カテゴリを追加</button>
      </div>
    </form>
  </div>

  <?php if ($cnt === 0): ?>
    <p>まだカテゴリがありません。</p>
  <?php else: ?>

  <h2>カテゴリ一覧</h2>
  <p>カテゴリ名やスラッグ、公開/非公開を変更したい場合は、カテゴリ名のテキストリンクをクリックして下さい。<br>
  カテゴリに設定する項目の追加・編集は、操作列にある<img src="assets/pen-solid.svg" alt="" style="width: 1.2rem;margin-right: 3px;">をクリックして下さい。</p>

  <input type="hidden" id="csrf_token" value="<?= h(csrf_get_token()) ?>">
  <div class="table-wrap">
    <table class="ta1">
      <thead>
        <tr>
          <th class="handle" aria-label="ドラッグで並び替え">☰</th>
          <th class="id">ID</th>
          <th class="category">カテゴリ名</th>
          <th class="slug">スラッグ</th>
          <th class="active">公開</th>
          <th class="date">作成日</th>
          <th class="btn-table">操作</th>
        </tr>
      </thead>
      <tbody id="cats-tbody">
        <?php foreach ($cats as $row): ?>
          <tr data-id="<?= (int)$row['id'] ?>">
            <td class="handle">☰</td>
            <td class="id"><?= (int)$row['id'] ?></td>
            <td class="category"><span class="cell-title"><a href="./category_edit.php?id=<?= (int)$row['id'] ?>"><?= h((string)$row['name']) ?></a></span></td>
            <td class="slug"><?= h((string)$row['slug']) ?></td>
            <td class="active"><?= ((int)$row['is_active'] === 1) ? '公開' : '非公開' ?></td>
            <td class="date"><?= h(substr((string)$row['created_at'], 0, 10)) ?></td>
            <td class="btn-table">
              <div class="btns">
                <a class="btn" href="./fields.php?category_id=<?= (int)$row['id'] ?>"><img src="assets/pen-solid.svg" alt="項目設計" title="項目設計"></a>
                <form method="post" action="./categories.php">
                <?= csrf_input_tag() ?>
                  <input type="hidden" name="_action" value="toggle">
                  <input type="hidden" name="id" value="<?= (int)$row['id'] ?>">
                  <input type="hidden" name="to_active" value="<?= ((int)$row['is_active'] === 1) ? 0 : 1 ?>">
                  <button type="submit" class="btn" title="<?= ((int)$row['is_active'] === 1) ? '非公開' : '公開' ?>">
                    <?php if ((int)$row['is_active'] === 1): ?>
                      <img src="assets/lock-open-solid.svg" alt="公開" title="公開">
                    <?php else: ?>
                      <img src="assets/lock-solid.svg" alt="非公開" title="非公開">
                    <?php endif; ?>
                  </button>
                </form>
                <form method="post" action="./categories.php" onsubmit="return confirm('このカテゴリを削除します。よろしいですか？※登録データも削除されるのでご注意下さい。');">
                <?= csrf_input_tag() ?>
                  <input type="hidden" name="_action" value="delete">
                  <input type="hidden" name="id" value="<?= (int)$row['id'] ?>">
                  <button type="submit" class="btn danger"><img src="assets/trash-can-solid.svg" alt="削除" title="削除"></button>
                </form>
              </div>
            </td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>
  <?php endif; ?>
</div>
<!--/#container-->

<script src="<?= h(asset_url('/admin/assets/admin.js', true)) ?>"></script>

<script>
(function(){
  const tbody = document.getElementById('cats-tbody');
  if (!tbody) return;

  // 行をドラッグ可能に
  Array.from(tbody.querySelectorAll('td.handle')).forEach(h => {
    h.setAttribute('draggable', 'true');
  });

  let dragging = null;

  tbody.addEventListener('dragstart', (e) => {
    const handle = e.target.closest('.handle');
    const row = e.target.closest('tr');
    if (!handle || !row) {
      e.preventDefault();
      return;
    }
    dragging = row;
    row.classList.add('dragging');
    e.dataTransfer.effectAllowed = 'move';
    e.dataTransfer.setData('text/plain', row.dataset.id || '');
  });

  tbody.addEventListener('dragend', () => {
    if (dragging) dragging.classList.remove('dragging');
    dragging = null;
  });

  tbody.addEventListener('dragover', (e) => {
    e.preventDefault();
    if (!dragging) return;
    const after = e.target.closest('tr');
    if (!after || after === dragging) return;

    const rect = after.getBoundingClientRect();
    const isAfter = (e.clientY - rect.top) > rect.height / 2;
    if (isAfter) after.after(dragging);
    else after.before(dragging);
  });

  tbody.addEventListener('drop', async (e) => {
    e.preventDefault();
    if (!dragging) return;

    const ids = Array.from(tbody.querySelectorAll('tr'))
      .map(tr => tr.getAttribute('data-id'))
      .filter(Boolean);

    try {
      const fd = new FormData();
      fd.append('_csrf', document.getElementById('csrf_token')?.value || '');
      fd.append('_action', 'sort');
      ids.forEach(id => fd.append('ids[]', id));

      const res = await fetch('./categories.php', { method: 'POST', body: fd, credentials: 'same-origin' });
      if (res.ok) location.reload();
    } catch (err) {
      console.error(err);
    }
  });
})();
</script>

</body>
</html>
