<?php
declare(strict_types=1);

require_once __DIR__ . '/../config.php';

/**
 * 管理者アカウントの読み書き＆検証ユーティリティ
 * 保管先: DATA_DIR/admin.json
 * フィールド: email, name, pass_hash, created_at, updated_at
 */

function admin_storage_path(): string {
    return DATA_DIR . '/admin.json';
}

function admin_exists(): bool {
    return is_file(admin_storage_path());
}

function admin_load(): ?array {
    $f = admin_storage_path();
    if (!is_file($f)) return null;
    $json = @file_get_contents($f);
    if ($json === false) return null;
    $data = json_decode($json, true);
    return (is_array($data) && isset($data['email'], $data['pass_hash'])) ? $data : null;
}

function admin_save(array $data): bool {
    $f = admin_storage_path();
    $json = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    if ($json === false) return false;
    if (@file_put_contents($f, $json, LOCK_EX) === false) return false;
    @chmod($f, 0664);
    return true;
}

/**
 * 初期管理者を作成（メール・パスワード必須）
 */
function admin_create(string $email, string $password, string $name = 'Admin'): void {
    $email = strtolower(trim($email));
    if ($email === '' || $password === '') {
        throw new InvalidArgumentException('メールアドレスとパスワードは必須です。');
    }
    $hash = password_hash($password, PASSWORD_DEFAULT);
    if ($hash === false) {
        throw new RuntimeException('パスワードのハッシュ化に失敗しました。');
    }
    $now = date('c');
    $data = [
        'email'      => $email,
        'name'       => $name,
        'pass_hash'  => $hash,
        'created_at' => $now,
        'updated_at' => $now,
    ];
    if (!admin_save($data)) {
        throw new RuntimeException('管理者情報の保存に失敗しました。');
    }
}

/**
 * 認証（メール＋パスワードの検証）
 */
function admin_verify(string $email, string $password): bool {
    $email = strtolower(trim($email));
    $data = admin_load();
    if (!$data) return false;
    if ($email !== strtolower((string)($data['email'] ?? ''))) return false;
    $hash = (string)($data['pass_hash'] ?? '');
    if ($hash === '' || !password_verify($password, $hash)) return false;
    return true;
}

/**
 * メール変更／パスワード変更（必要時）
 */
function admin_update_email(string $newEmail): void {
    $data = admin_load();
    if (!$data) throw new RuntimeException('管理者が未登録です。');
    $email = strtolower(trim($newEmail));
    if ($email === '') throw new InvalidArgumentException('メールアドレスが不正です。');
    $data['email'] = $email;
    $data['updated_at'] = date('c');
    if (!admin_save($data)) throw new RuntimeException('保存に失敗しました。');
}

function admin_update_password(string $newPassword): void {
    $data = admin_load();
    if (!$data) throw new RuntimeException('管理者が未登録です。');
    $hash = password_hash($newPassword, PASSWORD_DEFAULT);
    if ($hash === false) throw new RuntimeException('パスワードのハッシュ化に失敗しました。');
    $data['pass_hash']  = $hash;
    $data['updated_at'] = date('c');
    if (!admin_save($data)) throw new RuntimeException('保存に失敗しました。');
}
