<?php
declare(strict_types=1);

require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth/_auth.php';
require_once __DIR__ . '/../security/_csrf.php';

// 未ログインならログインへ
auth_require_login();

// 保存先（DATA_DIR/settings.json）
function tpcms_settings_path(): string {
  return DATA_DIR . '/settings.json';
}
function tpcms_settings_load(): array {
  $defaults = [
    'site_name'            => '',
    'admin_email'          => '',
    // ↓ 追加（フォーム設定）
    'admin_notify_subject' => '［{site}］お問い合わせ：{title} (#{id})',
    'auto_reply'           => false,
    'auto_reply_subject'   => '',
    'auto_reply_body'      => '',
    'mail_signature'       => '',
  ];
  $f = tpcms_settings_path();
  if (is_file($f)) {
    $json = @file_get_contents($f);
    if ($json !== false) {
      $a = json_decode($json, true);
      if (is_array($a)) {
        return [
          'site_name'            => (string)($a['site_name']            ?? $defaults['site_name']),
          'admin_email'          => (string)($a['admin_email']          ?? $defaults['admin_email']),
          'admin_notify_subject' => (string)($a['admin_notify_subject'] ?? $defaults['admin_notify_subject']),
          'auto_reply'           => (bool)  ($a['auto_reply']           ?? $defaults['auto_reply']),
          'auto_reply_subject'   => (string)($a['auto_reply_subject']   ?? $defaults['auto_reply_subject']),
          'auto_reply_body'      => (string)($a['auto_reply_body']      ?? $defaults['auto_reply_body']),
          'mail_signature'       => (string)($a['mail_signature']       ?? $defaults['mail_signature']),
        ];
      }
    }
  }
  return $defaults;
}
function tpcms_settings_save(array $a): bool {
  $data = [
    'site_name'            => (string)($a['site_name']            ?? ''),
    'admin_email'          => (string)($a['admin_email']          ?? ''),
    'admin_notify_subject' => (string)($a['admin_notify_subject'] ?? '［{site}］お問い合わせ：{title} (#{id})'),
    'auto_reply'           => (bool)  ($a['auto_reply']           ?? false),
    'auto_reply_subject'   => (string)($a['auto_reply_subject']   ?? ''),
    'auto_reply_body'      => (string)($a['auto_reply_body']      ?? ''),
    'mail_signature'       => (string)($a['mail_signature']       ?? ''),
    'updated_at'           => date('c'),
  ];
  $json = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
  if ($json === false) return false;
  if (!is_dir(DATA_DIR)) @mkdir(DATA_DIR, 0775, true);
  $ok = @file_put_contents(tpcms_settings_path(), $json, LOCK_EX) !== false;
  if ($ok) @chmod(tpcms_settings_path(), 0664);
  return $ok;
}

$err = '';
$ok  = (isset($_GET['ok']) && (string)$_GET['ok'] === '1');

// POST（保存）
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'POST') {
  csrf_check_or_die();

  // 基本
  $siteName   = trim((string)($_POST['site_name'] ?? ''));
  $adminEmail = trim((string)($_POST['admin_email'] ?? ''));

  // フォーム設定
  $adminNotifySubject = trim((string)($_POST['admin_notify_subject'] ?? ''));
  $autoReply          = isset($_POST['auto_reply']) && (string)$_POST['auto_reply'] === '1';
  $autoReplySubject   = trim((string)($_POST['auto_reply_subject'] ?? ''));
  $autoReplyBody      = (string)($_POST['auto_reply_body'] ?? ''); // 改行保持
  $mailSignature      = (string)($_POST['mail_signature'] ?? '');  // 改行保持

  // バリデーション
  if (mb_strlen($siteName, 'UTF-8') > 200) {
    $err = 'サイト名が長すぎます（200文字まで）。';
  } elseif ($adminEmail !== '' && !filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
    $err = '管理者メールの形式が正しくありません。';
  } elseif (mb_strlen($adminNotifySubject, 'UTF-8') > 200) {
    $err = '通知メール件名テンプレートが長すぎます（200文字まで）。';
  } elseif (mb_strlen($autoReplySubject, 'UTF-8') > 200) {
    $err = '自動返信の件名が長すぎます（200文字まで）。';
  } elseif (mb_strlen($autoReplyBody, 'UTF-8') > 10000) {
    $err = '自動返信の本文が長すぎます（10000文字まで）。';
  } elseif (mb_strlen($mailSignature, 'UTF-8') > 2000) {
    $err = '署名が長すぎます（2000文字まで）。';
  } else {
    $saveOk = tpcms_settings_save([
      'site_name'            => $siteName,
      'admin_email'          => $adminEmail,
      'admin_notify_subject' => $adminNotifySubject !== '' ? $adminNotifySubject : '［{site}］お問い合わせ：{title} (#{id})',
      'auto_reply'           => $autoReply,
      'auto_reply_subject'   => $autoReplySubject,
      'auto_reply_body'      => $autoReplyBody,
      'mail_signature'       => $mailSignature,
    ]);
    if ($saveOk) {
      header('Location: ' . url_for('/admin/settings.php', ['ok' => 1]), true, 302);
      exit;
    } else {
      $err = '保存に失敗しました。権限を確認してください。';
    }
  }
}

// 表示用
$st  = tpcms_settings_load();
$css = asset_url('/admin/assets/admin.css', true);
$js  = asset_url('/admin/assets/admin.js',  true);
?><!doctype html>
<html lang="ja">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>基本設定｜Template Party CMS</title>
  <link rel="stylesheet" href="<?= h($css) ?>">
</head>
<body class="settings">
<?php include __DIR__ . '/_header.php'; ?>

<div id="container">
  <h1>基本設定</h1>

  <?php if ($ok): ?>
    <p class="ok">保存しました。</p>
  <?php elseif ($err !== ''): ?>
    <p class="err"><?= h($err) ?></p>
  <?php endif; ?>

  <form action="./settings.php" method="post">
    <?= csrf_input_tag() ?>

      <h2>サイト基本情報</h2>
      <div class="row">
        <label class="label">サイト名
          <input type="text" class="input" name="site_name"
                 value="<?= h($st['site_name']) ?>" maxlength="200"
                 placeholder="例）サンプル不動産">
        </label>
      </div>
      <div class="row mb3rem">
        <label class="label">管理者メール（通知先）
          <input type="email" class="input" name="admin_email"
                 value="<?= h($st['admin_email']) ?>"
                 placeholder="admin@example.com" autocomplete="email">
        </label>
      </div>

      <h2>フォーム設定（問い合わせ）</h2>

      <div class="row">
        <label class="label">
          <input type="checkbox" name="auto_reply" value="1" <?= $st['auto_reply'] ? 'checked' : '' ?>>
          自動返信メールを送信する
        </label>
      </div>

      <div class="row">
        <label class="label">通知メールの件名
          <input type="text" class="input w100" name="admin_notify_subject"
                 value="<?= h($st['admin_notify_subject']) ?>" maxlength="200"
                 placeholder="［{site}］お問い合わせ：{title} (#{id})">
        </label>
        <span class="small">利用可能な置換：{site}, {title}, {id}, {category}</span>
      </div>

      <div class="row">
        <label class="label">自動返信の件名
          <input type="text" class="input w100" name="auto_reply_subject"
                 value="<?= h($st['auto_reply_subject']) ?>" maxlength="200"
                 placeholder="例）<?= h($st['site_name'] !== '' ? $st['site_name'] : 'サイト名') ?>：お問い合わせを受け付けました">
        </label>
      </div>

      <div class="row">
        <label class="label">自動返信の本文
          <textarea class="input w100" name="auto_reply_body" rows="6"
            placeholder="例）◯◯様&#10;&#10;お問い合わせありがとうございます。担当者より折り返しご連絡いたします。"><?= h($st['auto_reply_body']) ?></textarea>
        </label>
      </div>

      <div class="row">
        <label class="label">署名など
          <textarea class="input w100" name="mail_signature" rows="6"
            placeholder="例）株式会社サンプル&#10;サンプル不動産&#10;TEL: 00-0000-0000&#10;MAIL: info@example.com"><?= h($st['mail_signature']) ?></textarea>
        </label>
      </p>

    <p><button type="submit" class="btn1">保存する</button></p>
  </form>
</div>

<script src="<?= h($js) ?>"></script>
</body>
</html>
