<?php
declare(strict_types=1);

/**
 * Template Party CMS (tpcms) - 一覧・詳細特化型CMS
 * 初期ブート：共通ヘルパ
 *
 * 提供関数:
 * - h(mixed $s): string             HTMLエスケープ
 * - url_for(string $path, array $qs = []): string
 * - asset_url(string $path, bool $withVersion = false): string
 *
 * 依存: config.php（BASE_PATH/APP_ROOT）
 */

require_once __DIR__ . '/config.php';

/**
 * HTMLエスケープ（UTF-8固定）
 */
if (!function_exists('h')) {
    function h($s): string {
        return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    }
}

/**
 * 内部リンク生成（サブディレ配下でも正しいURLを返す）
 * - $path は "/list" や "admin/index.php" など。外部URL（http/https/mailto/tel/#）はそのまま返す
 * - $qs   はクエリ配列（http_build_queryで付与）
 */
if (!function_exists('url_for')) {
    function url_for(string $path = '', array $qs = []): string {
        $p = trim($path);

        // 外部 or 特殊スキームはそのまま
        if ($p === '' ||
            preg_match('~^[a-z][a-z0-9+.\-]*://~i', $p) || // http(s), ftp など
            str_starts_with($p, 'mailto:') ||
            str_starts_with($p, 'tel:') ||
            str_starts_with($p, '#')) {
            $url = $p;
        } else {
            // 先頭スラッシュ有無を吸収して BASE_PATH と連結
            $p = '/' . ltrim($p, '/');
            $base = (defined('BASE_PATH') && BASE_PATH !== '/') ? BASE_PATH : '';
            $url = $base . $p;
        }

        if ($qs) {
            $query = http_build_query($qs, '', '&', PHP_QUERY_RFC3986);
            $url .= (str_contains($url, '?') ? '&' : '?') . $query;
        }
        return $url;
    }
}

/**
 * アセットURL生成
 * - 相対パスを BASE_PATH と連結（外部URLはそのまま）
 * - $withVersion = true で、ローカル実体があれば mtime を v= に付加
 */
if (!function_exists('asset_url')) {
    function asset_url(string $path, bool $withVersion = false): string {
        // 外部URLはそのまま
        if (preg_match('~^[a-z][a-z0-9+.\-]*://~i', $path)) {
            return $path;
        }

        $url = url_for($path);

        if ($withVersion) {
            $fs = APP_ROOT . '/' . ltrim($path, '/');
            if (is_file($fs)) {
                $v = (string)@filemtime($fs);
                $url .= (str_contains($url, '?') ? '&' : '?') . 'v=' . rawurlencode($v);
            }
        }
        return $url;
    }
}

/**
 * 現在のテンプレートディレクトリを返す。
 * ・通常：APP_ROOT/templates
 * ・デモ等：環境変数 TPCMS_TEMPLATES_DIR が有効ならそちらを優先
 */
if (!function_exists('tpcms_templates_dir')) {
    function tpcms_templates_dir(): string {
        $env = getenv('TPCMS_TEMPLATES_DIR');
        if (is_string($env) && $env !== '' && is_dir($env)) {
            // 末尾スラッシュを詰める
            return rtrim($env, "/");
        }
        return APP_ROOT . '/templates';
    }
}
