<?php
declare(strict_types=1);

/**
 * tpcms ライト版レンダラ
 * ・{{BASE_PATH}} と {{CATEGORIES}} のみ展開
 * ・それ以外の {{...}} はそのまま（非展開）
 */

require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/front.php'; // front_categories_summary() を利用
require_once __DIR__ . '/macro.php'; // FEATURE_ITEMS を本流の実装で描画

/**
 * ライト版レンダラ
 * @param string $html templates/*.html の生HTML
 * @return string 展開後HTML（BASE_PATH / CATEGORIES のみ）
 */
function tpcms_render_light(string $html): string {
    // BASE_PATH：'/' のときは空、それ以外はそのまま
    $base = (defined('BASE_PATH') && BASE_PATH !== '/' && BASE_PATH !== '') ? BASE_PATH : '';

    // まず BASE_PATH を展開
    $out = str_replace('{{BASE_PATH}}', $base, $html);

    // {{CATEGORIES ...}} を属性付きで展開（許可：show_zero="0|1"）
    $out = preg_replace_callback('/\{\{\s*CATEGORIES(?:\s+([^}]*))?\}\}/i', function ($m) {
        // 属性パース（"…" と '…' に対応）
        $attrStr = isset($m[1]) ? (string)$m[1] : '';
        $attrs = [];
        if ($attrStr !== '') {
            if (preg_match_all('/([A-Za-z0-9_:-]+)\s*=\s*"(.*?)"|([A-Za-z0-9_:-]+)\s*=\s*\'(.*?)\'/u', $attrStr, $mm, PREG_SET_ORDER)) {
                foreach ($mm as $r) {
                    $key = strtolower((string)($r[1] !== '' ? $r[1] : $r[3]));
                    $val = (string)($r[2] !== '' ? $r[2] : $r[4]);
                    $attrs[$key] = $val;
                }
            }
        }
        $showZero = (isset($attrs['show_zero']) && (string)$attrs['show_zero'] === '1');

        // 公開カテゴリサマリ取得（slug, name, count）
        $cats = front_categories_summary(true);

        // ord 順に並べ替え（存在すれば）
        try {
            $ordMap = [];
            $st = db()->query('SELECT slug, COALESCE(ord, 999999) AS o FROM categories WHERE is_active = 1');
            foreach ($st->fetchAll(PDO::FETCH_ASSOC) ?: [] as $r) {
                $s = (string)($r['slug'] ?? '');
                if ($s !== '') $ordMap[$s] = (int)$r['o'];
            }
            if (!empty($ordMap)) {
                usort($cats, function ($a, $b) use ($ordMap) {
                    $sa = (string)($a['slug'] ?? ''); $sb = (string)($b['slug'] ?? '');
                    $oa = $ordMap[$sa] ?? 999999;      $ob = $ordMap[$sb] ?? 999999;
                    if ($oa === $ob) return strcmp($sa, $sb);
                    return $oa <=> $ob;
                });
            }
        } catch (\Throwable $e) { /* 失敗時は元順でOK */ }

        // <li>… を生成（show_zero=0 なら 0件を非表示）
        $lis = '';
        foreach ($cats as $c) {
            $slug = (string)($c['slug'] ?? ''); if ($slug === '') continue;
            $name = (string)($c['name'] ?? $slug);
            $cnt  = (int)($c['count'] ?? 0);
            if (!$showZero && $cnt <= 0) continue;
            $url  = url_for('/list', ['category' => $slug]);
            $lis .= '<li><a href="' . h($url) . '">' . h($name) . '<span class="badge">【' . (string)$cnt . '】</span></a></li>';
        }
        return $lis;
    }, $out);

    // {{FEATURE_ITEMS ...}} をライト版でも描画（許可：tpl="..." / layout="..."）
    $out = preg_replace_callback('/\{\{\s*FEATURE_ITEMS(?:\s+([^}]*))?\}\}/i', function ($m) {
        // 属性パース（"…" と '…' 両対応）
        $attrStr = isset($m[1]) ? (string)$m[1] : '';
        $attrs = [];
        if ($attrStr !== '') {
            if (preg_match_all('/([A-Za-z0-9_:-]+)\s*=\s*"(.*?)"|([A-Za-z0-9_:-]+)\s*=\s*\'(.*?)\'/u', $attrStr, $mm, PREG_SET_ORDER)) {
                foreach ($mm as $r) {
                    $key = strtolower((string)($r[1] !== '' ? $r[1] : $r[3]));
                    $val = (string)($r[2] !== '' ? $r[2] : $r[4]);
                    $attrs[$key] = $val;
                }
            }
        }
        // ctx は静的ページなので空でOK（特集は settings / DB を参照）
        return macro_feature_items([], $attrs);
    }, $out);

    return $out;
}
