<?php
declare(strict_types=1);

require_once __DIR__ . '/config.php';

/**
 * SQLite PDO を返す（DATA_DIR 配下に tpcms.sqlite3 を作成）
 * - 例: /data/tpcms.sqlite3
 * - 外部キー ON
 */
function db(): PDO {
    static $pdo = null;
    if ($pdo instanceof PDO) {
        return $pdo;
    }
    $dbPath = DATA_DIR . '/tpcms.sqlite3';
    $pdo = new PDO('sqlite:' . $dbPath, null, null, [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES   => false,
    ]);
    $pdo->exec('PRAGMA foreign_keys = ON');
    return $pdo;
}

/**
 * LD-02: カテゴリ用テーブルのマイグレーション
 * - categories(id PK, name, slug UNIQUE, is_active, created_at, updated_at)
 * - 並び順は created_at DESC を想定（インデックスは created_at に付与）
 */
function db_migrate_categories(): void {
    $stmts = [
        // テーブル本体
        "CREATE TABLE IF NOT EXISTS categories (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL,
            slug TEXT NOT NULL UNIQUE,
            is_active INTEGER NOT NULL DEFAULT 1,
            created_at TEXT NOT NULL DEFAULT (datetime('now')),
            updated_at TEXT
        )",
        // よく使う絞り込み・並びのためのインデックス
        "CREATE INDEX IF NOT EXISTS idx_categories_active ON categories(is_active)",
        "CREATE INDEX IF NOT EXISTS idx_categories_created_at ON categories(created_at)"
    ];
    $pdo = db();
    foreach ($stmts as $sql) {
        $pdo->exec($sql);
    }
}

/**
 * LD-03: fields テーブル（カテゴリごとの入力項目定義）
 *  - UNIQUE(category_id, "key") でカテゴリ内ユニークを保証
 *  - 真偽系は 0/1、options は JSON文字列（TEXT）で保持
 */
function db_migrate_fields(): void {
    $pdo = db();
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS fields (
            id           INTEGER PRIMARY KEY AUTOINCREMENT,
            category_id  INTEGER NOT NULL,
            label        TEXT    NOT NULL,
            \"key\"      TEXT    NOT NULL,  -- SQLite予約語回避のためダブルクォート
            type         TEXT    NOT NULL,  -- text/textarea/select/radio/checkbox/number/currency/date/assets/map_iframe
            required     INTEGER NOT NULL DEFAULT 0,  -- 0/1（配布版では未使用）
            searchable   INTEGER NOT NULL DEFAULT 0,  -- 0/1（検索UI/検索ロジック用）
            list_visible INTEGER NOT NULL DEFAULT 0,  -- 0/1（管理一覧用）
            list_site    INTEGER NOT NULL DEFAULT 0,  -- 0/1（サイト側LIST用）
            detail_site  INTEGER NOT NULL DEFAULT 0,  -- 0/1（サイト側DETAIL用）
            sortable     INTEGER NOT NULL DEFAULT 0,  -- 0/1（一部型のみ意味あり）
            unit         TEXT    DEFAULT NULL,
            options      TEXT    DEFAULT NULL,        -- JSON（選択肢や数値min/max/step 等）
            ord          INTEGER NOT NULL DEFAULT 0,  -- 表示順
            FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE
        );
    ");
    // ユニーク＆よく使う並びのためのインデックス
    $pdo->exec('CREATE UNIQUE INDEX IF NOT EXISTS idx_fields_category_key ON fields(category_id, "key");');
    $pdo->exec('CREATE INDEX IF NOT EXISTS idx_fields_category_ord ON fields(category_id, ord);');
    // 既存DBに新カラムを追加（存在すれば例外→握りつぶし）
    try { $pdo->exec('ALTER TABLE fields ADD COLUMN list_site   INTEGER NOT NULL DEFAULT 0'); } catch (Throwable $e) {}
    try { $pdo->exec('ALTER TABLE fields ADD COLUMN detail_site INTEGER NOT NULL DEFAULT 0'); } catch (Throwable $e) {}
}

/**
 * LD-11: settings テーブル（サイト設定のKVS）
 *  - settings(key TEXT PRIMARY KEY, value TEXT NOT NULL DEFAULT '')
 */
function db_migrate_settings(): void {
    $pdo = db();
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS settings (
            key   TEXT PRIMARY KEY,
            value TEXT NOT NULL DEFAULT ''
        );
    ");
}
