<?php
declare(strict_types=1);

/**
 * 認証ユーティリティ
 * - 未ログイン時に管理画面へアクセスした場合は login.php へ誘導
 * - ログイン成功時は session_regenerate_id(true)
 *
 * 依存: helpers.php（→ config.php でセッション開始/BASE_PATH定義済み）
 */

require_once __DIR__ . '/../helpers.php';

/**
 * 現在のログイン情報を返す（未ログインなら null）
 */
if (!function_exists('auth_current_user')) {
    function auth_current_user(): ?array {
        $u = $_SESSION['auth_user'] ?? null;
        return (is_array($u) && isset($u['id'])) ? $u : null;
    }
}

/**
 * ログインしているか判定
 */
if (!function_exists('auth_is_logged_in')) {
    function auth_is_logged_in(): bool {
        return auth_current_user() !== null;
    }
}

/**
 * ログイン成功処理（ログイン時に呼び出す）
 * 例）auth_login_success(['id'=>1, 'email'=>'a@b', 'name'=>'Admin']);
 */
if (!function_exists('auth_login_success')) {
    function auth_login_success(array $user): void {
        // セッションIDを再発行（固定化攻撃対策）
        if (session_status() === PHP_SESSION_ACTIVE) {
            session_regenerate_id(true);
        }
        $_SESSION['auth_user'] = [
            'id'    => $user['id']    ?? null,
            'email' => $user['email'] ?? '',
            'name'  => $user['name']  ?? '',
            'time'  => time(),
        ];
    }
}

/**
 * ログアウト処理
 */
if (!function_exists('auth_logout')) {
    function auth_logout(): void {
        unset($_SESSION['auth_user']);
        if (session_status() === PHP_SESSION_ACTIVE) {
            session_regenerate_id(true);
        }
    }
}

/**
 * 未ログインならログイン画面へリダイレクトして終了
 * - ?return= に現在のパスを付与（BASE_PATH から始まる安全な相対先のみ許可）
 */
if (!function_exists('auth_require_login')) {
    function auth_require_login(): void {
        if (auth_is_logged_in()) return;

        // 現在のパス（クエリ含む）を return に埋め込む
        $uri = (string)($_SERVER['REQUEST_URI'] ?? '/');
        // 管理画面での利用を想定：必ず BASE_PATH から始まる
        $return = $uri !== '' ? $uri : (BASE_PATH === '/' ? '/' : BASE_PATH . '/');

        $loginUrl = url_for('/admin/login.php', ['return' => $return]);
        header('Location: ' . $loginUrl, true, 302);
        exit;
    }
}

/**
 * ?return= パラメータに指定された復帰先へ、安全にリダイレクト
 * - BASE_PATH から始まらない値は破棄してデフォルト（/admin/）へ
 */
if (!function_exists('auth_redirect_back_or_default')) {
    function auth_redirect_back_or_default(string $default = '/admin/'): void {
        $ret = $_GET['return'] ?? '';
        $ret = is_string($ret) ? $ret : '';

        $ok = false;
        if ($ret !== '') {
            // 絶対URL/スキーム禁止、BASE_PATH から始まる相対のみ
            if (!preg_match('~^[a-z][a-z0-9+.\-]*://~i', $ret)) {
                $ok = str_starts_with($ret, (BASE_PATH === '/' ? '/' : BASE_PATH . '/'));
            }
        }

        $to = $ok ? $ret : url_for($default);
        header('Location: ' . $to, true, 302);
        exit;
    }
}
