<?php
declare(strict_types=1);

/**
 * Template Party CMS (tpcms) - 一覧・詳細特化型CMS
 * 初期ブート：設定/共通
 *
 * 目的（LD-01より）:
 * - サブディレ対応（BASE_PATH）
 * - APP_NAME / PATHS / セッション名（cookie_path=BASE_PATH）
 * - 最小限のセキュリティヘッダ
 *
 * BASE_PATH は自動検出し、環境変数 TPCMS_BASE_PATH があればそれを優先します。
 */

// ------------------------------
// APP 基本
// ------------------------------
define('APP_NAME', 'tpcms'); // 配布名（LD-01）
// 浮動小数のJSON出力桁を最小化（例：0.1 が 0.1000... にならないように）
@ini_set('serialize_precision', '-1');
@ini_set('precision', '14');

define('APP_ROOT', realpath(__DIR__) ?: __DIR__); // プロジェクト実ディレクトリ

// ------------------------------
// BASE_PATH（サブディレ対応）
// 例）Web ルート直下: "/" / サブディレ: "/mysite"
// ------------------------------
(function () {
    $env = getenv('TPCMS_BASE_PATH');
    if (is_string($env) && $env !== '') {
        $bp = '/' . ltrim($env, '/');
        $bp = rtrim($bp, '/');
        define('BASE_PATH', $bp === '' ? '/' : $bp);
        return;
    }

    // 自動検出：DOCUMENT_ROOT と __DIR__ の相対で算出（管理画面/フロントどちらからでも同じ値）
    $doc = $_SERVER['DOCUMENT_ROOT'] ?? '';
    $doc = is_string($doc) ? realpath($doc) ?: '' : '';
    $app = realpath(__DIR__) ?: __DIR__;

    if ($doc !== '' && str_starts_with(str_replace('\\', '/', $app), str_replace('\\', '/', (string)$doc))) {
        $rel = substr(str_replace('\\', '/', $app), strlen(str_replace('\\', '/', (string)$doc)));
        $rel = '/' . ltrim($rel, '/');
        $rel = rtrim($rel, '/');
        define('BASE_PATH', $rel === '' ? '/' : $rel);
    } else {
        // フォールバック：スクリプト配置ディレクトリから
        $dir = dirname((string)($_SERVER['SCRIPT_NAME'] ?? '')) ?: '/';
        // 管理画面(/admin/配下)からでもプロジェクト直下を指したい場合は必要に応じて調整
        define('BASE_PATH', rtrim($dir, '/') === '' ? '/' : rtrim($dir, '/'));
    }
})();

// ------------------------------
// ストレージパス（DIR）
// - 原本は非公開（UPLOAD_DIR）
// - 公開サムネ等は PUBLIC_UPLOAD_DIR（ドメイン公開下）
// ------------------------------
define('DATA_DIR', APP_ROOT . '/data');
define('UPLOAD_DIR', APP_ROOT . '/uploads');              // 非公開原本置き場
define('PUBLIC_UPLOAD_DIR', APP_ROOT . '/public_uploads'); // 公開用派生ファイル置き場

// ------------------------------
// セッション/クッキー
// ------------------------------
// cookie_path は BASE_PATH に固定（LD-01要件）
$secure = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off';
session_set_cookie_params([
    'lifetime' => 0,
    'path'     => BASE_PATH === '/' ? '/' : BASE_PATH,
    'secure'   => $secure,
    'httponly' => true,
    'samesite' => 'Lax',
]);
session_name('tpcms_sess');
if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}

// ------------------------------
// 最小セキュリティヘッダ（ヘッダ未送信時のみ）
// ※ CSP は後段で導入
// ------------------------------
if (!headers_sent()) {
    header('X-Content-Type-Options: nosniff');
    header('Referrer-Policy: same-origin');
    header('X-Frame-Options: SAMEORIGIN');
}

// ------------------------------
// 便利：実行時に必要なディレクトリが無ければ作成（権限は環境に合わせて調整）
// ------------------------------
foreach ([DATA_DIR, UPLOAD_DIR] as $dir) {
    if (!is_dir($dir)) {
        @mkdir($dir, 0775, true);
    }
}
// PUBLIC_UPLOAD_DIR は将来のサムネイル生成などの処理内で、必要時に遅延作成します。

// ---------- 検索設定：checkbox の結合方式（同一フィールド内） ----------
// 'OR' = どれか一致（既定） / 'AND' = 全て一致
if (!defined('TPCMS_CHECKBOX_JOIN')) {
    define('TPCMS_CHECKBOX_JOIN', 'OR');
}
