<?php
declare(strict_types=1);

require_once __DIR__ . '/../db.php';

/**
 * LD-03 フィールド定義モデル（下地）
 * - 読み込まれた時点で fields テーブルを保証
 * - 以後のステップでCRUDやプリセット投入を実装
 */
db_migrate_fields();

/**
 * options を型に応じて正規化して JSON 文字列を返す（保存用）。適さない場合は null。
 * - select/radio/checkbox: 1行1項目。形式「value|label」。label省略時は value を流用。
 * - number/currency: {"min":0,"max":999,"step":1} のようなJSON。数値のみ許可。
 * - それ以外: null（保存しない）
 */
function fields_normalize_options(string $type, string $raw): ?string {
    $raw = trim($raw);
    if ($raw === '') return null;

    if ($type === 'select' || $type === 'radio' || $type === 'checkbox') {
        $lines = preg_split("/\r\n|\r|\n/u", $raw);
        $out = [];
        foreach ($lines as $ln) {
            $ln = trim((string)$ln);
            if ($ln === '') continue;
            if (strpos($ln, '|') !== false) {
                [$v, $l] = explode('|', $ln, 2);
                $v = trim($v);
                $l = trim($l);
                if ($v === '') continue;
                $out[] = ['value' => $v, 'label' => ($l !== '' ? $l : $v)];
            } else {
                $out[] = ['value' => $ln, 'label' => $ln];
            }
        }
        return empty($out) ? null : json_encode($out, JSON_UNESCAPED_UNICODE);
    }

    if ($type === 'number' || $type === 'currency') {
        $j = json_decode($raw, true);
        if (!is_array($j)) return null;
        $out = [];
        foreach (['min','max','step'] as $k) {
            if (isset($j[$k]) && (is_int($j[$k]) || is_float($j[$k]) || (is_string($j[$k]) && is_numeric($j[$k])))) {
                $out[$k] = 0 + $j[$k];
            }
        }
        return empty($out) ? null : json_encode($out, JSON_UNESCAPED_UNICODE);
    }

    return null;
}

/**
 * 保存済み options（JSON or NULL）を「編集用の人間可読テキスト」に戻す。
 * - select/radio/checkbox: 1行1項目（value|label）へ展開
 * - number/currency: {"min":0,"max":999,"step":1} をキーのみ整形して返す
 * - それ以外: そのまま（文字列 or 空）
 */
function fields_options_to_human(string $type, ?string $jsonOrText): string {
    $src = (string)($jsonOrText ?? '');
    if ($src === '') return '';

    if ($type === 'select' || $type === 'radio' || $type === 'checkbox') {
        $arr = json_decode($src, true);
        if (!is_array($arr)) return $src;
        $lines = [];
        foreach ($arr as $opt) {
            $v = isset($opt['value']) ? (string)$opt['value'] : '';
            $l = isset($opt['label']) ? (string)$opt['label'] : '';
            if ($v === '') continue;
            $lines[] = ($l !== '' && $l !== $v) ? ($v.'|'.$l) : $v;
        }
        return implode("\n", $lines);
    }

    if ($type === 'number' || $type === 'currency') {
        $j = json_decode($src, true);
        if (!is_array($j)) return '';
        $out = [];
        foreach (['min','max','step'] as $k) {
            if (array_key_exists($k, $j) && (is_int($j[$k]) || is_float($j[$k]) || (is_string($j[$k]) && is_numeric($j[$k])))) {
                $out[$k] = 0 + $j[$k];
            }
        }
        return empty($out) ? '' : json_encode($out, JSON_UNESCAPED_UNICODE);
    }

    return $src;
}

/**
 * 指定カテゴリの全フィールドを取得（ord→id順）
 */
function fields_fetch_all(int $categoryId): array {
    $pdo = db();
    $st = $pdo->prepare('SELECT id, category_id, label, "key", type, required, searchable, list_visible, sortable, unit, options, ord
                         FROM fields
                         WHERE category_id = ?
                         ORDER BY ord ASC, id ASC');
    $st->execute([$categoryId]);
    return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

/**
 * 後続で使うメタをまとめて返す
 * - fields: 全行（将来のフォーム自動描画や編集に利用）
 * - list_columns: 一覧表示するキーの配列（list_visible=1）
 * - searchable: 検索対象キーの配列（searchable=1）
 * - sortable: 並べ替え可能キーの配列（sortable=1）
 */
function fields_meta_for_category(int $categoryId): array {
    $rows = fields_fetch_all($categoryId);
    $list = [];
    $search = [];
    $sort = [];
    foreach ($rows as $r) {
        if (!empty($r['list_visible'])) $list[] = (string)$r['key'];
        if (!empty($r['searchable']))   $search[] = (string)$r['key'];
        if (!empty($r['sortable']))     $sort[] = (string)$r['key'];
    }
    return [
        'fields'       => $rows,
        'list_columns' => $list,
        'searchable'   => $search,
        'sortable'     => $sort,
    ];
}
